<?php
/**
 * Rubik_Elementor_Widgets class.
 *
 * @category   Class
 * @package    RubikElementorWidgets
 * @subpackage WordPress
 * @author     Mostafa Kalantari Fard <design3nter@gmail.com>
 * @copyright  2022 Mostafa Kalantari Fard
 * @license    https://opensource.org/licenses/GPL-3.0 GPL-3.0-only
 * @link       link(https://www.rtl-theme.com/,
 *             Rubik Elementor Widgets)
 * @since      1.0.0
 * php version 7.3.9
 */

if ( ! defined( 'ABSPATH' ) ) {
    // Exit if accessed directly.
    exit;
}

/**
 * Main Elementor Rubik_Elementor_Widgets Class
 *
 * The init class that runs the Rubik Elementor Widgets plugin.
 * Intended To make sure that the plugin's minimum requirements are met.
 */
final class Rubik_Elementor_Widgets {

    /**
     * Plugin Version
     *
     * @since 1.0.0
     * @var string The plugin version.
     */
    const VERSION = '1.0.0';

    /**
     * Minimum Elementor Version
     *
     * @since 1.0.0
     * @var string Minimum Elementor version required to run the plugin.
     */
    const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

    /**
     * Minimum PHP Version
     *
     * @since 1.0.0
     * @var string Minimum PHP version required to run the plugin.
     */
    const MINIMUM_PHP_VERSION = '7.0';

    /**
     * Constructor
     *
     * @since 1.0.0
     * @access public
     */
    public function __construct() {
        // Load the translation.
        add_action( 'init', array( $this, 'i18n' ) );

        // Initialize the plugin.
        add_action( 'plugins_loaded', array( $this, 'init' ) );
    }

    /**
     * Load Textdomain
     *
     * Load plugin localization files.
     * Fired by `init` action hook.
     *
     * @since 1.0.0
     * @access public
     */
    public function i18n() {
        load_plugin_textdomain( 'rubik-elementor-widgets' );
    }

    /**
     * Initialize the plugin
     *
     * Validates that Elementor is already loaded.
     * Checks for basic plugin requirements, if one check fail don't continue,
     * if all check have passed include the plugin class.
     *
     * Fired by `plugins_loaded` action hook.
     *
     * @since 1.0.0
     * @access public
     */
    public function init() {

        // Check if Elementor installed and activated.
        if ( ! did_action( 'elementor/loaded' ) ) {
            add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
            return;
        }

        // Check for required Elementor Pro version.
        if ( ! function_exists( 'is_plugin_active' ) )
            require_once( ABSPATH . '/wp-admin/includes/plugin.php' );

        if ( !is_plugin_active('elementor-pro/elementor-pro.php') ) {
            add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
            return;
        }


        // Check for required PHP version.
        if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
            add_action( 'admin_notices', array( $this, 'admin_notice_minimum_php_version' ) );
            return;
        }

        // Once we get here, We have passed all validation checks so we can safely include our widgets.
        require_once 'class-widgets.php';
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have Elementor installed or activated.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_missing_main_plugin() {
        deactivate_plugins( plugin_basename( RUBIK_ELEMENTOR_WIDGETS ) );
        unset($_GET['activate']);
        $message = '<div class="notice notice-warning is-dismissible"><p>برای فعالسازی این افزونه، ابتدا باید افزونه المنتور پرو را نصب و فعال کنید.</p></div>';
        echo $message;
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required PHP version.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_minimum_php_version() {
        deactivate_plugins( plugin_basename( RUBIK_ELEMENTOR_WIDGETS ) );

        $message = sprintf(
            wp_kses(
                '<div class="notice notice-warning is-dismissible"><p><strong>"%1$s"</strong> requires <strong>"%2$s"</strong> version %3$s or greater.</p></div>',
                array(
                    'div' => array(
                        'class'  => array(),
                        'p'      => array(),
                        'strong' => array(),
                    ),
                )
            ),
            'Rubik Elementor Widgets',
            'Elementor',
            self::MINIMUM_ELEMENTOR_VERSION
        );
        echo $message;
    }
}

// Instantiate Rubik_Elementor_Widgets.
new Rubik_Elementor_Widgets();
